package com.zov.maps.integration.demo.zovMaps

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.os.IBinder
import com.zov.maps.ipc.ZovMapsIpcStyles
import com.zov.maps.ipc.models.callback.AidlException
import com.zov.maps.ipc.models.callback.AidlResult
import com.zov.maps.ipc.models.callback.AsyncCallback
import com.zov.maps.ipc.models.callback.StringData
import com.zov.maps.ipc.models.styles.ListStylesMapper
import com.zov.maps.ipc.models.styles.ZovMapObjectStyle
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class ZovMapsIpcStylesHelper(
    private val ctx: Context,
) {

    var connected: Boolean = false
        private set

    fun checkAndConnect() {
        if (!connected) {
            connect()
        }
    }

    fun connect() {
        createExplicitIntent()?.also {
            ctx.bindService(
                it, serviceConnection, Context.BIND_AUTO_CREATE + Context.BIND_IMPORTANT
            )
            connected = true
        }
    }

    fun disconnect() {
        if (connected) {
            ctx.unbindService(serviceConnection)
        }
        connected = false
    }

    private var ipc: ZovMapsIpcStyles? = null

    private var serviceConnection = object : ServiceConnection {

        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            ipc = ZovMapsIpcStyles.Stub.asInterface(service)
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            ipc = null
        }
    }

    private fun createExplicitIntent(): Intent? {
        val intent = Intent(IPC_NAME)
        val services = ctx.packageManager.queryIntentServices(intent, 0)
        if (services.isEmpty()) {
            return null
        }
        return Intent(intent).apply {
            val resolveInfo = services[0]
            val packageName = resolveInfo.serviceInfo.packageName
            val className = resolveInfo.serviceInfo.name
            component = ComponentName(packageName, className)
        }
    }

    suspend fun getStyle(id: String) = suspendCancellableCoroutine<ZovMapObjectStyle?> { continuation ->
        val callback = object : AsyncCallback.Stub() {

            override fun onSuccess(aidlResult: AidlResult<*>) {
                val data = aidlResult.data as? ZovMapObjectStyle
                continuation.resume(data)
            }

            override fun onError(aidlException: AidlException) {
                continuation.resumeWithException(UnsupportedOperationException(aidlException.errorMessage))
            }
        }
        ipc!!.getStyle(id, callback)
    }

    suspend fun getAllStyles() = suspendCancellableCoroutine<List<ZovMapObjectStyle>> { continuation ->
        val callback = object : AsyncCallback.Stub() {

            override fun onSuccess(aidlResult: AidlResult<*>) {
                val stylesJson = (aidlResult.data as StringData).value
                val styles = ListStylesMapper.mapFrom(stylesJson).styles
                continuation.resume(styles)
            }

            override fun onError(aidlException: AidlException) {
                continuation.resumeWithException(UnsupportedOperationException(aidlException.errorMessage))
            }
        }
        ipc!!.getAllStyles(callback)
    }

    private companion object {

        const val IPC_NAME = "com.zov.maps.ipc.ZovMapsIpcStyles"
    }
}
