package com.zov.maps.integration.demo.zovMaps

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.os.IBinder
import com.zov.maps.ipc.ZovMapsIpcLayers
import com.zov.maps.ipc.models.callback.AidlException
import com.zov.maps.ipc.models.callback.AidlResult
import com.zov.maps.ipc.models.callback.AsyncCallback
import com.zov.maps.ipc.models.layers.ZovGraphicLayer
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class ZovMapsIpcLayersHelper(
    private val ctx: Context,
) {

    var connected: Boolean = false
        private set

    fun checkAndConnect() {
        if (!connected) {
            connect()
        }
    }

    fun connect() {
        createExplicitIntent()?.also {
            ctx.bindService(
                it, serviceConnection, Context.BIND_AUTO_CREATE + Context.BIND_IMPORTANT
            )
            connected = true
        }
    }

    fun disconnect() {
        if (connected) {
            ctx.unbindService(serviceConnection)
        }
        connected = false
    }

    private var ipc: ZovMapsIpcLayers? = null

    private var serviceConnection = object : ServiceConnection {

        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            ipc = ZovMapsIpcLayers.Stub.asInterface(service)
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            ipc = null
        }
    }

    private fun createExplicitIntent(): Intent? {
        val intent = Intent(IPC_NAME)
        val services = ctx.packageManager.queryIntentServices(intent, 0)
        if (services.isEmpty()) {
            return null
        }
        return Intent(intent).apply {
            val resolveInfo = services[0]
            val packageName = resolveInfo.serviceInfo.packageName
            val className = resolveInfo.serviceInfo.name
            component = ComponentName(packageName, className)
        }
    }

    fun saveGraphicLayer(layer: ZovGraphicLayer) {
        ipc!!.saveGraphicLayer(layer)
    }

    fun deleteLayer(id: String) {
        ipc!!.delete(id)
    }

    fun deleteObjects(id: String) {
        ipc!!.deleteObjects(id)
    }

    suspend fun getLayer(id: String) = suspendCancellableCoroutine<ZovGraphicLayer?> { continuation ->
        val callback = object : AsyncCallback.Stub() {

            override fun onSuccess(aidlResult: AidlResult<*>) {
                val data = aidlResult.data as? ZovGraphicLayer
                continuation.resume(data)
            }

            override fun onError(aidlException: AidlException) {
                continuation.resumeWithException(UnsupportedOperationException(aidlException.errorMessage))
            }
        }
        ipc!!.getLayer(id, callback)
    }

    private companion object {

        const val IPC_NAME = "com.zov.maps.ipc.ZovMapsIpcLayers"
    }
}
