package com.zov.maps.integration.demo.zovMaps

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.os.IBinder
import com.zov.maps.ipc.ZovMapsIpcGeo
import com.zov.maps.ipc.models.callback.AidlException
import com.zov.maps.ipc.models.callback.AidlResult
import com.zov.maps.ipc.models.callback.AsyncCallback
import com.zov.maps.ipc.models.objects.ZovMapCircle
import com.zov.maps.ipc.models.objects.ZovMapLine
import com.zov.maps.ipc.models.objects.ZovMapObjectInfo
import com.zov.maps.ipc.models.objects.ZovMapPoint
import com.zov.maps.ipc.models.objects.ZovMapPolygon
import com.zov.maps.ipc.models.objects.ZovMapRectangle
import com.zov.maps.ipc.models.objects.ZovMapSector
import com.zov.maps.ipc.models.objects.ZovMapText
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class ZovMapsIpcGeoHelper(
    private val ctx: Context,
) {

    var connected: Boolean = false
        private set

    fun checkAndConnect() {
        if (!connected) {
            connect()
        }
    }

    fun connect() {
        createExplicitIntent()?.also {
            ctx.bindService(
                it, serviceConnection, Context.BIND_AUTO_CREATE + Context.BIND_IMPORTANT
            )
            connected = true
        }
    }

    fun disconnect() {
        if (connected) {
            ctx.unbindService(serviceConnection)
        }
        connected = false
    }

    private var ipc: ZovMapsIpcGeo? = null

    private var serviceConnection = object : ServiceConnection {

        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            ipc = ZovMapsIpcGeo.Stub.asInterface(service)
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            ipc = null
        }
    }

    private fun createExplicitIntent(): Intent? {
        val intent = Intent(IPC_NAME)
        val services = ctx.packageManager.queryIntentServices(intent, 0)
        if (services.isEmpty()) {
            return null
        }
        return Intent(intent).apply {
            val resolveInfo = services[0]
            val packageName = resolveInfo.serviceInfo.packageName
            val className = resolveInfo.serviceInfo.name
            component = ComponentName(packageName, className)
        }
    }

    fun saveText(text: ZovMapText) {
        ipc!!.saveText(text)
    }

    fun savePoint(point: ZovMapPoint) {
        ipc!!.savePoint(point)
    }

    fun saveLine(line: ZovMapLine) {
        ipc!!.saveLine(line)
    }

    fun savePolygon(polygon: ZovMapPolygon) {
        ipc!!.savePolygon(polygon)
    }

    fun saveRectangle(rectangle: ZovMapRectangle) {
        ipc!!.saveRectangle(rectangle)
    }

    fun saveCircle(circle: ZovMapCircle) {
        ipc!!.saveCircle(circle)
    }

    fun saveSector(sector: ZovMapSector) {
        ipc!!.saveSector(sector)
    }

    fun deleteObject(id: String) {
        ipc!!.delete(id)
    }

    suspend fun getObject(id: String) = suspendCancellableCoroutine<ZovMapObjectInfo?> { continuation ->
        val callback = object : AsyncCallback.Stub() {

            override fun onSuccess(aidlResult: AidlResult<*>) {
                val data = aidlResult.data as? ZovMapObjectInfo
                continuation.resume(data)
            }

            override fun onError(aidlException: AidlException) {
                continuation.resumeWithException(UnsupportedOperationException(aidlException.errorMessage))
            }
        }
        ipc!!.getObject(id, callback)
    }

    private companion object {

        const val IPC_NAME = "com.zov.maps.ipc.ZovMapsIpcGeo"
    }
}
