package com.zov.maps.integration.demo.utils.viewmodel

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import com.zov.maps.integration.demo.utils.livedata.SingleLiveData
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlin.coroutines.CoroutineContext

abstract class BaseViewModel(
    protected val mainContext: CoroutineContext,
    protected val bgContext: CoroutineContext,
) : ViewModel() {

    protected val _msg = SingleLiveData<String>()
    val msg: LiveData<String>
        get() = _msg

    protected val _finish = SingleLiveData<Boolean>()
    val finish: LiveData<Boolean>
        get() = _finish

    protected val _progress = MutableLiveData<Boolean>()
    val progress: LiveData<Boolean>
        get() = _progress

    protected fun CoroutineScope.launchWithProgress(
        doWork: suspend () -> Unit,
        onError: (e: Throwable) -> Unit = {},
        onFinally: () -> Unit = {},
        progressTimeout: Long = 250L,
        context: CoroutineContext = bgContext,
    ): Job = launch(context) {
        val progressJob = launch {
            delay(progressTimeout)
            _progress.postValue(true)
        }
        try {
            doWork()
        } catch (e: Throwable) {
            onError(e)
        } finally {
            progressJob.cancel()
            _progress.postValue(false)
            onFinally()
        }
    }

    protected fun CoroutineScope.launchWithOutProgress(
        doWork: suspend () -> Unit,
        onError: (e: Throwable) -> Unit = {},
        onFinally: () -> Unit = {},
        context: CoroutineContext = bgContext,
    ): Job = launch(context) {
        try {
            doWork()
        } catch (e: Throwable) {
            onError(e)
        } finally {
            onFinally()
        }
    }
}
