package com.zov.maps.integration.demo.ui.stylesIpc

import android.content.Context
import androidx.core.content.ContextCompat
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import com.zov.maps.integration.demo.R
import com.zov.maps.integration.demo.ui.stylesIpc.sort.StylesSorter
import com.zov.maps.integration.demo.utils.color.ColorParser
import com.zov.maps.integration.demo.utils.viewmodel.BaseViewModel
import com.zov.maps.integration.demo.zovMaps.ZovMapsIpcStylesHelper
import com.zov.maps.ipc.models.styles.StyleType
import com.zov.maps.ipc.models.styles.ZovMapObjectStyle
import kotlin.coroutines.CoroutineContext

class StylesIpcViewModel(
    private val ctx: Context,
    private val stylesSorter: StylesSorter,
    private val zovMapsIpcStylesHelper: ZovMapsIpcStylesHelper,
    mainContext: CoroutineContext,
    bgContext: CoroutineContext
) : BaseViewModel(mainContext, bgContext), DefaultLifecycleObserver {

    private val _stylesUI = MutableLiveData<List<StyleItemUI>>()
    val stylesUI: LiveData<List<StyleItemUI>>
        get() = _stylesUI

    override fun onResume(owner: LifecycleOwner) {
        zovMapsIpcStylesHelper.checkAndConnect()
    }

    override fun onStop(owner: LifecycleOwner) {
        zovMapsIpcStylesHelper.disconnect()
    }

    fun loadStyles() {
        viewModelScope.launchWithProgress(
            doWork = {
                val styles = zovMapsIpcStylesHelper.getAllStyles()
                val sortedStyles = stylesSorter.sortStyles(styles)
                val uiItems = convert(sortedStyles)
                _stylesUI.postValue(uiItems)
            },
            onError = { error ->
                _msg.postValue("Проверьте установлено и запущено ли приложение `ZOV Карты` ${error.message}")
            },
        )
    }

    fun loadStyle() {
        viewModelScope.launchWithProgress(
            doWork = {
                val existingStyle = "faf7a238-cb3e-4446-821b-84875ee59301"
                val noStyle = "none"
                val style = zovMapsIpcStylesHelper.getStyle(existingStyle) ?: run {
                    _msg.postValue("Нет стиля с таким id")
                    _stylesUI.postValue(emptyList())
                    return@launchWithProgress
                }
                val sortedStyles = stylesSorter.sortStyles(listOf(style))
                val uiItems = convert(sortedStyles)
                _stylesUI.postValue(uiItems)
            },
            onError = { error ->
                _msg.postValue("Проверьте установлено и запущено ли приложение `ZOV Карты` ${error.message}")
            },
        )
    }

    private fun convert(list: List<ZovMapObjectStyle>): List<StyleItemUI> {
        return list.map { style ->
            val styleType = StyleType.find(style.typeId)
            val iconColor = style.color?.takeIf { it.isNotBlank() }?.let { ColorParser.parse(it) }
            val icon = when (styleType) {
                StyleType.TEXT -> {
                    ContextCompat.getDrawable(ctx, R.drawable.ic_text)?.mutate()?.apply {
                        iconColor?.also { setTint(it) }
                    }
                }
                StyleType.POINT -> {
                    ContextCompat.getDrawable(ctx, R.drawable.ic_point)
                }
                StyleType.LINE -> {
                    ContextCompat.getDrawable(ctx, R.drawable.ic_line)?.mutate()?.apply {
                        iconColor?.also { setTint(it) }
                    }
                }
                StyleType.POLYGON -> {
                    ContextCompat.getDrawable(ctx, R.drawable.ic_polygon)?.mutate()?.apply {
                        iconColor?.also { setTint(it) }
                    }
                }
                else -> null
            }
            StyleItemUI(id = style.id, name = style.name, category = style.category.takeIf { it.isNotBlank() }?.let {
                "Категория: $it"
            } ?: "Без категории", subCategory = style.subCategory.takeIf { it.isNotBlank() }?.let {
                "Раздел: $it"
            } ?: "Без раздела", description = style.description.takeIf { it.isNotBlank() }?.let {
                "Описание: $it"
            } ?: "Без описания", icon = icon, type = styleType?.name ?: "UNKNOWN")
        }
    }

}
