package com.zov.maps.integration.demo.ui.layersIpc

import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.asLiveData
import androidx.lifecycle.viewModelScope
import com.zov.maps.integration.demo.ui.common.IdLayerDispatcher
import com.zov.maps.integration.demo.utils.identifier.IdGenerator
import com.zov.maps.integration.demo.utils.viewmodel.BaseViewModel
import com.zov.maps.integration.demo.zovMaps.ZovMapsIpcLayersHelper
import com.zov.maps.ipc.models.layers.ZovGraphicLayer
import kotlin.coroutines.CoroutineContext

class LayersIpcViewModel(
    private val zovMapsLayersIpcHelper: ZovMapsIpcLayersHelper,
    private val idGenerator: IdGenerator,
    private val idLayerDispatcher: IdLayerDispatcher,
    mainContext: CoroutineContext,
    bgContext: CoroutineContext,
) : BaseViewModel(mainContext, bgContext), DefaultLifecycleObserver {

    val idLayerLiveData: LiveData<String> = idLayerDispatcher.flow.asLiveData()
    val idLayer: String
        get() = idLayerDispatcher.value

    private var counter: Int = 0

    override fun onResume(owner: LifecycleOwner) {
        zovMapsLayersIpcHelper.checkAndConnect()
    }

    override fun onStop(owner: LifecycleOwner) {
        zovMapsLayersIpcHelper.disconnect()
    }

    fun saveGraphicLayer() {
        try {
            counter++
            val layer = ZovGraphicLayer(
                id = idLayer,
                name = "From Integration Demo: $counter",
            )
            zovMapsLayersIpcHelper.saveGraphicLayer(layer)
            _msg.value = "Сохранен слой\n${layer.name}"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun deleteLayer() {
        try {
            zovMapsLayersIpcHelper.deleteLayer(idLayer)
            _msg.value = "Слой удален"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun deleteObjects() {
        try {
            zovMapsLayersIpcHelper.deleteObjects(idLayer)
            _msg.value = "Удалены объекты слоя"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun generateNewLayerId() {
        idLayerDispatcher.tryEmit(idGenerator.random())
    }

    fun loadLayer() {
        viewModelScope.launchWithProgress(
            doWork = {
                val layer = zovMapsLayersIpcHelper.getLayer(idLayer) ?: run {
                    _msg.postValue("Нет слоя с таким id")
                    return@launchWithProgress
                }
                _msg.postValue(layer.toString())
            },
            onError = { error ->
                when (error) {
                    is UnsupportedOperationException -> {
                        _msg.postValue(error.message ?: "UnsupportedOperationException")
                    }
                    else -> {
                        _msg.postValue("Проверьте установлено и запущено ли приложение `ZOV Карты` ${error.message}")
                    }
                }
            },
        )
    }
}
