package com.zov.maps.integration.demo.ui.geoIpc

import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.asLiveData
import androidx.lifecycle.viewModelScope
import com.zov.maps.integration.demo.ui.common.IdLayerDispatcher
import com.zov.maps.integration.demo.ui.common.IdObjectDispatcher
import com.zov.maps.integration.demo.utils.identifier.IdGenerator
import com.zov.maps.integration.demo.utils.time.TimeProvider
import com.zov.maps.integration.demo.utils.viewmodel.BaseViewModel
import com.zov.maps.integration.demo.zovMaps.ZovMapsIpcGeoHelper
import com.zov.maps.ipc.models.base.ZovMapGeoPoint
import com.zov.maps.ipc.models.objects.ZovMapCircle
import com.zov.maps.ipc.models.objects.ZovMapLine
import com.zov.maps.ipc.models.objects.ZovMapPoint
import com.zov.maps.ipc.models.objects.ZovMapPolygon
import com.zov.maps.ipc.models.objects.ZovMapRectangle
import com.zov.maps.ipc.models.objects.ZovMapSector
import com.zov.maps.ipc.models.objects.ZovMapText
import com.zov.maps.ipc.utils.toGeometry
import kotlin.coroutines.CoroutineContext

class GeoIpcViewModel(
    private val zovMapsIpcGeoHelper: ZovMapsIpcGeoHelper,
    private val timeProvider: TimeProvider,
    private val idGenerator: IdGenerator,
    private val idLayerDispatcher: IdLayerDispatcher,
    private val idObjectDispatcher: IdObjectDispatcher,
    mainContext: CoroutineContext,
    bgContext: CoroutineContext,
) : BaseViewModel(mainContext, bgContext), DefaultLifecycleObserver {

    val idLayerLiveData: LiveData<String> = idLayerDispatcher.flow.asLiveData()
    val idLayer: String
        get() = idLayerDispatcher.value

    val idObjectLiveData: LiveData<String> = idObjectDispatcher.flow.asLiveData()
    val idObject: String
        get() = idObjectDispatcher.value

    private var counter: Int = 0
    private val createdTime = timeProvider.currentTimeMillis
    private val userName = "IPC Demo User"

    override fun onResume(owner: LifecycleOwner) {
        zovMapsIpcGeoHelper.checkAndConnect()
    }

    override fun onStop(owner: LifecycleOwner) {
        zovMapsIpcGeoHelper.disconnect()
    }

    fun saveText() {
        try {
            counter++
            val text = ZovMapText(
                id = idObject,
                idLayer = idLayer,
                idStyle = "cd2af08d-be20-493b-8354-547e3772c22e",
                geometry = ZovMapGeoPoint(62.0, 32.0).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                userName = userName,
                text = "text_$counter",
                label = "label_$counter",
                description = "description_$counter",
                rotationDegrees = 45.0,
                isMoveMapView = false,
            )
            zovMapsIpcGeoHelper.saveText(text)
            _msg.value = "Сохранен текст"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun savePoint() {
        try {
            counter++
            val point = ZovMapPoint(
                id = idObject,
                idLayer = idLayer,
                idStyle = "9ba0a8dd-5525-4b1b-89eb-c67918ad6485",
                geometry = ZovMapGeoPoint(63.0, 33.0).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                label = "label_$counter",
                description = "description_$counter",
                userName = userName,
                rotationDegrees = 315.0,
            )
            zovMapsIpcGeoHelper.savePoint(point)
            _msg.value = "Сохранена точка"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun saveLine() {
        try {
            counter++
            val line = ZovMapLine(
                id = idObject,
                idLayer = idLayer,
                idStyle = "7759f351-d7b5-4206-8f25-0605c8c3c3b5",
                geometry = listOf(
                    ZovMapGeoPoint(60.0, 33.0),
                    ZovMapGeoPoint(59.0, 34.0),
                    ZovMapGeoPoint(60.0, 35.0),
                ).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                label = "label_$counter",
                description = "description_$counter",
                userName = userName,
            )
            zovMapsIpcGeoHelper.saveLine(line)
            _msg.value = "Сохранена линия"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun savePolygon() {
        try {
            counter++
            val polygon = ZovMapPolygon(
                id = idObject,
                idLayer = idLayer,
                idStyle = "9018403e-5eaf-40f9-b848-956cdc3ee412",
                geometry = listOf(
                    ZovMapGeoPoint(60.0, 26.0),
                    ZovMapGeoPoint(61.0, 27.0),
                    ZovMapGeoPoint(60.0, 29.0),
                    ZovMapGeoPoint(60.0, 26.0),
                ).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                label = "label_$counter",
                description = "description_$counter",
                userName = userName,
            )
            zovMapsIpcGeoHelper.savePolygon(polygon)
            _msg.value = "Сохранен полигон"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun saveRectangle() {
        try {
            counter++
            val rectangle = ZovMapRectangle(
                id = idObject,
                idLayer = idLayer,
                idStyle = "9018403e-5eaf-40f9-b848-956cdc3ee412",
                geometry = listOf(
                    ZovMapGeoPoint(58.0, 33.0),
                    ZovMapGeoPoint(58.0, 34.0),
                    ZovMapGeoPoint(57.0, 34.0),
                    ZovMapGeoPoint(57.0, 33.0),
                    ZovMapGeoPoint(58.0, 33.0),
                ).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                label = "label_$counter",
                description = "description_$counter",
                userName = userName,
            )
            zovMapsIpcGeoHelper.saveRectangle(rectangle)
            _msg.value = "Сохранен прямоугольник"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun saveCircle() {
        try {
            counter++
            val circle = ZovMapCircle(
                id = idObject,
                idLayer = idLayer,
                idStyle = "9018403e-5eaf-40f9-b848-956cdc3ee412",
                geometry = listOf(
                    ZovMapGeoPoint(58.0, 30.0),
                    ZovMapGeoPoint(57.0, 31.0),
                ).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                label = "label_$counter",
                description = "description_$counter",
                userName = userName,
            )
            zovMapsIpcGeoHelper.saveCircle(circle)
            _msg.value = "Сохранен круг"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun saveSector() {
        try {
            counter++
            val sector = ZovMapSector(
                id = idObject,
                idLayer = idLayer,
                idStyle = "9018403e-5eaf-40f9-b848-956cdc3ee412",
                geometry = listOf(
                    ZovMapGeoPoint(60.5, 31.5),
                    ZovMapGeoPoint(62.0, 30.0),
                ).toGeometry(),
                createTime = createdTime,
                lastEditTime = timeProvider.currentTimeMillis,
                label = "label_$counter",
                description = "description_$counter",
                userName = userName,
                angle = 55.0,
            )
            zovMapsIpcGeoHelper.saveSector(sector)
            _msg.value = "Сохранен сектор"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun deleteObject() {
        try {
            zovMapsIpcGeoHelper.deleteObject(idObject)
            _msg.value = "Объект удален"
        } catch (e: Exception) {
            _msg.value = "Проверьте установлено и запущено ли приложение `ZOV Карты`"
        }
    }

    fun generateNewObjectId() {
        idObjectDispatcher.tryEmit(idGenerator.random())
    }

    fun loadObject() {
        viewModelScope.launchWithProgress(
            doWork = {
                val obj = zovMapsIpcGeoHelper.getObject(idObject) ?: run {
                    _msg.postValue("Нет объекта с таким id")
                    return@launchWithProgress
                }
                _msg.postValue(obj.toString())
            },
            onError = { error ->
                _msg.postValue("Проверьте установлено и запущено ли приложение `ZOV Карты` ${error.message}")
            },
        )
    }
}
