import contextlib
import datetime as dt
import json
import os
import random
import sys
import time
from pathlib import Path
from typing import Union

import requests
from eth_typing import Hash32, HexStr
from hexbytes import HexBytes
from web3 import Web3
from web3.eth import Eth
from web3.types import TxReceipt

import constants
import enums
from logger import logging
from starknet_py.contract import Contract
from starknet_py.net.account.account import Account
from starknet_py.net.gateway_client import GatewayClient
from starknet_py.net.models import chain_from_network
from starknet_py.net.signer.stark_curve_signer import KeyPair, StarkCurveSigner


def int_hash_to_hex(hast_int: int, hash_lenght: int = 64) -> str:
    hash_hex = hex(hast_int)[2:]
    hash_hex = hash_hex.rjust(hash_lenght, '0')
    return f'0x{hash_hex}'


def get_account(
    network: str,
    private_key: str,
    address: str,
    proxy: dict[str, str] = None,
    signer_class=None
) -> Account:
    client = GatewayClient(
        network,
        proxy=proxy if proxy is None else proxy['http'],
        user_agent='Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/116.0.0.0 Safari/537.36'
    )

    key_pair = KeyPair.from_private_key(
        key=private_key
    )

    if signer_class is None:
        signer_class = StarkCurveSigner

    signer = signer_class(
        address,
        key_pair,
        chain_from_network(client.net)
    )

    return Account(
        client=client,
        address=address,
        signer=signer
    )


def get_starknet_contract(
    address: str,
    abi: list,
    provider: Account
) -> Contract:
    return Contract(
        address=address,
        abi=abi,
        provider=provider
    )


def get_starknet_erc20_contract(
    token_address: str,
    provider: Account
) -> Contract:
    with open(Path(__file__).parent / 'abi' / 'STARKNET_ERC20.json') as file:
        erc20_abi = json.load(file)

    return get_starknet_contract(
        address=token_address,
        abi=erc20_abi,
        provider=provider
    )


def sleep(sleep_time: float):
    logging.info(f'[Sleep] Sleeping for {round(sleep_time, 2)} seconds. If you want to skip this, press Ctrl+C')
    try:
        time.sleep(sleep_time)
    except KeyboardInterrupt:
        logging.info('[Sleep] Skipping sleep')


def random_sleep():
    min_sleep_time = getattr(random_sleep, 'min_sleep_time', 1)
    max_sleep_time = getattr(random_sleep, 'max_sleep_time', 10)
    sleep_time = round(random.uniform(min_sleep_time, max_sleep_time), 2)
    sleep(sleep_time)


def estimate_message_fee(
    client: GatewayClient,
    from_address: str,
    to_address: str,
    entry_point_selector: str,
    payload: list[str],
    proxy: dict[str, str] = None
) -> dict[str, Union[int, str]]:
    response = requests.post(
        url=f'{client._feeder_gateway_client.url}/estimate_message_fee?blockNumber=pending',
        json={
            'entry_point_selector': entry_point_selector,
            'from_address': from_address,
            'payload': payload,
            'to_address': to_address
        },
        proxies=proxy
    )

    if response.status_code != 200:
        logging.error(f'[Estimate Message Fee] Failed to estimate message fee: {response.text}')
        return

    return response.json()


def suggest_gas_fees(
    network_name: enums.NetworkNames,
    proxy: dict[str, str] = None
):
    last_update = getattr(suggest_gas_fees, 'last_update', dt.datetime.fromtimestamp(0))
    last_network = getattr(suggest_gas_fees, 'network_name', None)
    if dt.datetime.now() - last_update > dt.timedelta(seconds=10) or last_network != network_name:
        try:
            response = requests.get(
                url=f'https://gas-api.metaswap.codefi.network/networks/{network_name.value}/suggestedGasFees',
                proxies=proxy
            )
        except Exception:
            logging.error(f'[Gas] Failed to get gas price for {network_name.value}')
            return None
        else:
            if response.status_code != 200:
                logging.error(f'[Gas] Failed to get gas price for {network_name.value}: {response.text}')
                return None
            gas_json = response.json()
            medium_gas = gas_json['medium']
            gas_price = {
                'maxFeePerGas': Web3.to_wei(medium_gas['suggestedMaxFeePerGas'], 'gwei'),
                'maxPriorityFeePerGas': Web3.to_wei(medium_gas['suggestedMaxPriorityFeePerGas'], 'gwei')
            }
            suggest_gas_fees.gas_price = gas_price
            suggest_gas_fees.last_update = dt.datetime.now()
            suggest_gas_fees.network_name = network_name
            return gas_price
    else:
        return getattr(suggest_gas_fees, 'gas_price', None)


def wait_for_transaction_receipt(
    web3: Eth,
    txn_hash: Hash32 | HexBytes | HexStr,
    timeout: int = 300,
    logging_prefix: str = 'Receipt'
) -> TxReceipt:
    try:
        receipt = web3.wait_for_transaction_receipt(
            transaction_hash=txn_hash,
            timeout=timeout
        )
    except Exception as e:
        answer = input(f'[{logging_prefix}] Failed to get transaction receipt. Press Enter when transaction will be processed')
        try:
            receipt = web3.wait_for_transaction_receipt(
                transaction_hash=txn_hash,
                timeout=5
            )
        except Exception as e:
            logging.error(f'[{logging_prefix}] Failed to get transaction receipt: {e}')
            return None

    return receipt


@contextlib.contextmanager
def suppress_print():
    original_stdout = sys.stdout
    with open(os.devnull, 'w') as devnull:
        sys.stdout = devnull
        try:
            yield
        finally:
            sys.stdout = original_stdout


def test_proxy(proxy: dict[str, str]) -> bool:
    try:
        response = requests.get(
            url='https://google.com',
            proxies=proxy,
            timeout=5
        )
    except Exception:
        return False
    else:
        return True


def get_gas_price(
    private_key: str,
    network_name: enums.NetworkNames,
    proxy: dict[str, str] = None
) -> float:
    if network_name in {None, enums.NetworkNames.Exchange}:
        return 0

    if network_name in {
        enums.NetworkNames.ETH,
        enums.NetworkNames.Goerli,
        enums.NetworkNames.Starknet,
        enums.NetworkNames.Goerli
    }:
        if network_name in {
            enums.NetworkNames.ETH,
            enums.NetworkNames.Starknet
        }:
            network_name = enums.NetworkNames.ETH
        else:
            network_name = enums.NetworkNames.Goerli

        module_network = constants.NETWORKS[network_name]

        web3 = Web3(
            Web3.HTTPProvider(
                module_network.rpc_url,
                request_kwargs={
                    'proxies': proxy
                }
            )
        )
        gas_wei = int(web3.eth.gas_price)

        return float(Web3.from_wei(gas_wei, 'gwei'))

    return 0


def get_eth_price(default_price: float = 1850) -> float:
    last_update = getattr(get_eth_price, 'last_update', dt.datetime.fromtimestamp(0))
    if dt.datetime.now() - last_update > dt.timedelta(minutes=1):
        try:
            response = requests.get('https://api.coingecko.com/api/v3/simple/price?ids=ethereum&vs_currencies=usd')
            eth_price = response.json()['ethereum']['usd']
        except Exception as e:
            logging.error(f'[ETH Price] Failed to get ETH price: {e}. Setting to default (${default_price})')
            eth_price = default_price
        get_eth_price.eth_price = eth_price
        get_eth_price.last_update = dt.datetime.now()
    return get_eth_price.eth_price


def usd_to_eth(usd: float) -> float:
    eth_price = get_eth_price()
    return usd / eth_price


def str_to_felt(text: str) -> int:
    return int.from_bytes(text.encode(), 'big')


def extend_hex(hex_str: str | int, length: int) -> str:
    if isinstance(hex_str, int):
        hex_str = hex(hex_str)
    return hex_str.replace('0x', '0x' + '0' * (length - len(hex_str) + 2))
