import pytest

from starknet_py.common import create_compiled_contract
from starknet_py.net.models import StarknetChainId
from starknet_py.net.models.transaction import Declare, DeployAccount, Invoke
from starknet_py.net.signer.stark_curve_signer import KeyPair, StarkCurveSigner
from starknet_py.tests.e2e.fixtures.constants import CONTRACTS_COMPILED_DIR
from starknet_py.tests.e2e.fixtures.misc import read_contract

compiled_contract = read_contract(
    "erc20_compiled.json", directory=CONTRACTS_COMPILED_DIR
)


@pytest.mark.parametrize(
    "transaction",
    [
        Invoke(
            sender_address=0x1,
            calldata=[1, 2, 3],
            max_fee=10000,
            signature=[],
            nonce=23,
            version=1,
        ),
        DeployAccount(
            class_hash=0x1,
            contract_address_salt=0x2,
            constructor_calldata=[1, 2, 3, 4],
            max_fee=10000,
            signature=[],
            nonce=23,
            version=1,
        ),
        Declare(
            contract_class=create_compiled_contract(
                compiled_contract=compiled_contract
            ),
            sender_address=123,
            max_fee=10000,
            signature=[],
            nonce=23,
            version=1,
        ),
    ],
)
def test_sign_transaction(transaction):
    signer = StarkCurveSigner(
        account_address=0x1,
        key_pair=KeyPair.from_private_key(0x1),
        chain_id=StarknetChainId.TESTNET,
    )

    signature = signer.sign_transaction(transaction)

    assert isinstance(signature, list)
    assert len(signature) > 0
    assert all(isinstance(i, int) for i in signature)
    assert all(i != 0 for i in signature)


def test_key_pair():
    key_pair = KeyPair(public_key="0x123", private_key="0x456")

    assert isinstance(key_pair.public_key, int)
    assert isinstance(key_pair.private_key, int)

    key_pair = KeyPair.from_private_key("0x789")

    assert isinstance(key_pair.public_key, int)
    assert isinstance(key_pair.private_key, int)
