from typing import List

from starknet_py.hash.address import compute_address
from starknet_py.hash.transaction import compute_deploy_account_transaction_hash
from starknet_py.hash.utils import compute_hash_on_elements, message_signature
from starknet_py.net.account.account import Account
from starknet_py.net.account.account_deployment_result import AccountDeploymentResult
from starknet_py.net.client_models import TransactionFinalityStatus
from starknet_py.net.models import StarknetChainId, DeployAccount
from starknet_py.net.signer.stark_curve_signer import KeyPair, StarkCurveSigner

import constants
import create_accounts
import enums
import utils
from logger import logging

NETWORK_TO_CHAIN = {
    enums.NetworkNames.Starknet: StarknetChainId.MAINNET,
    enums.NetworkNames.StarknetTestnet: StarknetChainId.TESTNET
}


class BraavosSigner(StarkCurveSigner):
    def _sign_deploy_account_transaction(self, transaction: DeployAccount) -> List[int]:
        contract_address = compute_address(
            salt=transaction.contract_address_salt,
            class_hash=transaction.class_hash,
            constructor_calldata=transaction.constructor_calldata,
            deployer_address=0,
        )

        txn_hash = compute_deploy_account_transaction_hash(
            version=transaction.version,
            contract_address=contract_address,
            class_hash=transaction.class_hash,
            constructor_calldata=transaction.constructor_calldata,
            max_fee=transaction.max_fee,
            nonce=transaction.nonce,
            salt=transaction.contract_address_salt,
            chain_id=self.chain_id
        )

        r, s = message_signature(
            msg_hash=compute_hash_on_elements(
                [txn_hash, create_accounts.ACCOUNT_CLASS_HASHES[enums.WalletNames.Braavos], 0, 0, 0, 0, 0, 0, 0]
            ),
            priv_key=self.private_key
        )

        return [r, s, create_accounts.ACCOUNT_CLASS_HASHES[enums.WalletNames.Braavos], 0, 0, 0, 0, 0, 0, 0]


async def deploy_wallet(
    private_key: str,
    address: str,
    network_name: enums.NetworkNames,
    wallet_name: enums.WalletNames,
    proxy: dict[str, str] = None
):
    logging.info(f'[Deploy Account] Deploying account for {wallet_name} wallet')

    network = constants.NETWORKS[network_name]

    account = utils.get_account(
        network=network.rpc_url,
        private_key=private_key,
        address=address,
        proxy=proxy,
        signer_class=BraavosSigner if wallet_name == enums.WalletNames.Braavos else None
    )

    balance_in_wei = await account.get_balance(
        constants.NETWORK_TOKENS[network_name, enums.TokenNames.ETH].int_contract_address
    )

    key_pair = KeyPair.from_private_key(private_key)

    chain = NETWORK_TO_CHAIN[network_name]

    constructor_calldata = create_accounts.get_constructor_calldata(
        wallet_name=wallet_name,
        key_pair=key_pair
    )

    try:
        if wallet_name == enums.WalletNames.Braavos:
            deploy_account_tx = await account.sign_deploy_account_transaction(
                class_hash=create_accounts.PROXY_CLASS_HASHES[wallet_name],
                contract_address_salt=key_pair.public_key,
                constructor_calldata=constructor_calldata,
                nonce=0,
                max_fee=None,
                auto_estimate=True,
            )

            if deploy_account_tx.max_fee > balance_in_wei:
                logging.critical('[Deploy Account] Insufficient balance to deploy account')
                return enums.TransactionStatus.INSUFFICIENT_BALANCE

            result = await account.client.deploy_account(deploy_account_tx)

            result = AccountDeploymentResult(
                hash=result.transaction_hash,
                account=account,
                _client=account.client
            )
        else:
            if balance_in_wei == 0:
                logging.critical('[Deploy Account] Insufficient balance to deploy account')
                return enums.TransactionStatus.INSUFFICIENT_BALANCE

            result = await Account.deploy_account(
                address=address,
                class_hash=create_accounts.PROXY_CLASS_HASHES[wallet_name],
                salt=key_pair.public_key,
                key_pair=key_pair,
                client=account.client,
                chain=chain,
                constructor_calldata=constructor_calldata,
                auto_estimate=True
            )
    except BaseException as e:
        if 'is unavailable for deployment' in str(e):
            logging.info('[Deploy Account] Account is already deployed')
            return enums.TransactionStatus.SUCCESS
        elif 'Not enough tokens at the specified address to cover deployment costs' in str(e):
            logging.critical('[Deploy Account] Insufficient balance to deploy account')
            return enums.TransactionStatus.INSUFFICIENT_BALANCE
        logging.error(f'[Deploy Account] Failed to deploy account: {e}')
        return enums.TransactionStatus.FAILED

    logging.info(f'[Deploy Account] Transaction: {network.txn_explorer_url}{utils.int_hash_to_hex(result.hash)}')

    receipt = await result.wait_for_acceptance()

    if receipt.status == TransactionFinalityStatus.NOT_RECEIVED:
        logging.error('[Deploy Account] Failed to deploy account')
        return enums.TransactionStatus.FAILED
    else:
        logging.info('[Deploy Account] Successfully deployed account')
        return enums.TransactionStatus.SUCCESS
